<?php
/**
 * Adcaver Ad Helper
 *
 * This plugin assists publishers in integrating with the Adcaver ad network.
 * It provides a settings page to manage verification codes, site IDs,
 * and enable/disable specific ad formats served by Adcaver.
 *
 * @package           Adcaver_Ad_Helper
 *
 * @wordpress-plugin
 * Plugin Name:       Adcaver Ad Helper
 * Plugin URI:        https://adcaver.com
 * Description:       Easily integrate your WordPress site with the Adcaver ad network. Manage verification and enable ad formats.
 * Version:           1.0.3
 * Author:            Edu Brazeal,Adcaver Team
 * Author URI:        https://edubrazeal.com
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       adcaver-ad-helper
 * Domain Path:       /languages
 * Requires at least: 5.8
 * Requires PHP:      7.2
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Define Core Plugin Constants
 */
define( 'ADCAVER_HELPER_VERSION', '1.0.3' );
define( 'ADCAVER_HELPER_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'ADCAVER_HELPER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'ADCAVER_HELPER_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
define( 'ADCAVER_HELPER_SETTINGS_SLUG', 'adcaver-ad-helper-settings' );
define( 'ADCAVER_HELPER_OPTION_NAME', 'adcaver_helper_settings' );
define( 'ADCAVER_SERVER_URL', 'https://adcaver.com' );

/**
 * Load the main plugin class responsible for orchestrating hooks.
 */
require_once ADCAVER_HELPER_PLUGIN_DIR . 'includes/class-adcaver-helper-loader.php';

/**
 * The code that runs during plugin activation.
 */
function activate_adcaver_ad_helper() {
	if ( false === get_option( ADCAVER_HELPER_OPTION_NAME ) ) {
		$default_settings = array(
			'site_id'                         => '',
			'verification_code'               => '',
			'auto_ad_posts_before_content'    => 'disabled',
			'auto_ad_posts_after_p1'          => 'disabled',
			'auto_ad_posts_after_p2'          => 'disabled',
			'auto_ad_posts_after_p3'          => 'disabled',
			'auto_ad_posts_mid_content'       => 'disabled',
			'auto_ad_posts_after_content'     => 'disabled',
			'auto_ad_pages_before_content'    => 'disabled',
			'auto_ad_pages_after_content'     => 'disabled',
			'auto_ad_homepage_before_content' => 'disabled',
			'auto_ad_homepage_after_content'  => 'disabled',
			'enable_vignette'                 => 0,
			'enable_anchor'                   => 0,
			'enable_sticky_bar'               => 0,
		);
		update_option( ADCAVER_HELPER_OPTION_NAME, $default_settings );
	}
}

/**
 * The code that runs during plugin deactivation.
 */
function deactivate_adcaver_ad_helper() {
	// Optional: Add deactivation cleanup if needed in the future.
}

register_activation_hook( __FILE__, 'activate_adcaver_ad_helper' );
register_deactivation_hook( __FILE__, 'deactivate_adcaver_ad_helper' );

/**
 * Retrieves the saved verification code and prints the meta tag in the <head>.
 *
 * This function is hooked into 'wp_head'.
 */
function adcaver_add_verification_tag_to_head() {
	
	// Get all the saved settings for your plugin
	$options = get_option( ADCAVER_HELPER_OPTION_NAME ); 

	// Check if the 'verification_code' field is set and is not empty
	if ( ! empty( $options['verification_code'] ) ) {
		
		// Sanitize the code for the 'content' attribute
		$verification_code = esc_attr( $options['verification_code'] );

		// This is the meta tag name your main plugin (Zeal Ad Manager) is looking for.
		$meta_tag_name = 'zeal-ad-verification';

		// Print the full meta tag to the head
		echo "\n\n";
		echo "<meta name=\"{$meta_tag_name}\" content=\"{$verification_code}\">\n";
	}
}

/**
 * Register the function to run on the 'wp_head' action.
 */
add_action( 'wp_head', 'adcaver_add_verification_tag_to_head' );

/**
 * Attempts to purge site-wide caches when the plugin settings are saved.
 *
 * This ensures that external verifiers (like the main plugin) receive
 * the fresh page with the new verification tag.
 *
 * @param mixed $old_value The old option value.
 * @param mixed $new_value The new option value.
 */
function adcaver_purge_caches_on_settings_save( $old_value, $new_value ) {

	// Check if the verification code has actually changed.
	$old_code = isset( $old_value['verification_code'] ) ? $old_value['verification_code'] : '';
	$new_code = isset( $new_value['verification_code'] ) ? $new_value['verification_code'] : '';

	if ( $old_code === $new_code ) {
		// Don't purge if the code hasn't changed.
		return;
	}

	// WP Rocket
	if ( function_exists( 'rocket_clean_domain' ) ) {
		rocket_clean_domain();
	}

	// W3 Total Cache
	if ( function_exists( 'w3tc_pgcache_flush' ) ) {
		w3tc_pgcache_flush();
	}

	// WP Super Cache
	if ( function_exists( 'wp_cache_clear_cache' ) ) {
		wp_cache_clear_cache();
	}

	// LiteSpeed Cache
	if ( method_exists( 'LiteSpeed_Cache_API', 'purge_all' ) ) {
		LiteSpeed_Cache_API::purge_all();
	}
	
	// WP Fastest Cache
	if ( function_exists( 'wpfc_clear_all_cache' ) ) {
		wpfc_clear_all_cache( true );
	}

	// SiteGround Optimizer
	if ( function_exists( 'sg_cachepress_purge_cache' ) ) {
		sg_cachepress_purge_cache();
	}

	// Kinsta
	if ( class_exists( '\Kinsta\Cache' ) && method_exists( '\Kinsta\Cache', 'purge_cache_full' ) ) {
		\Kinsta\Cache::purge_cache_full();
	}

	// WP Engine
	if ( class_exists( 'WPECommon' ) && method_exists( 'WPECommon', 'purge_memcached' ) ) {
		WPECommon::purge_memcached();
	}
	if ( class_exists( 'WPECommon' ) && method_exists( 'WPECommon', 'purge_varnish_cache' ) ) {
		WPECommon::purge_varnish_cache();
	}
    
    // Default WordPress Object Cache
    if ( function_exists( 'wp_cache_flush' ) ) {
        wp_cache_flush();
    }
}

/**
 * Register the cache purge function to run when our settings are updated.
 */
add_action( 'update_option_' . ADCAVER_HELPER_OPTION_NAME, 'adcaver_purge_caches_on_settings_save', 10, 2 );


/**
 * Begins execution of the plugin.
 *
 * @since    1.0.0
 */
function run_adcaver_ad_helper() {
	$plugin = new Adcaver_Helper_Loader();
	$plugin->run();
}

run_adcaver_ad_helper();