<?php
/**
 * Adcaver Ad Helper Admin
 *
 * Handles the admin-specific functionality, including the settings page menu item.
 *
 * @package    Adcaver_Ad_Helper
 * @subpackage Adcaver_Ad_Helper/admin
 * @author     Edu Brazeal, Adcaver Team <https://edubrazeal.com>
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

class Adcaver_Helper_Admin {

	/**
	 * Adds the plugin's settings page to the WordPress admin menu under "Settings".
	 *
	 * @since    1.0.0
	 */
	public function add_plugin_admin_menu() {
		add_options_page(
			__( 'Adcaver Ad Helper Settings', 'adcaver-ad-helper' ), // Page title
			__( 'Adcaver Ads', 'adcaver-ad-helper' ),                // Menu title
			'manage_options',                                        // Capability required
			ADCAVER_HELPER_SETTINGS_SLUG,                            // Menu slug
			array( $this, 'display_settings_page' )                  // Callback function to render the page
		);
	}

	/**
	 * Renders the settings page content.
	 *
	 * Includes the view file which contains the HTML and Settings API functions.
	 *
	 * @since    1.0.0
	 */
	public function display_settings_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Include the settings page view
		require_once ADCAVER_HELPER_PLUGIN_DIR . 'admin/views/settings-page.php';
	}

	/**
	 * Enqueues stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 * @param    string $hook_suffix The current admin page hook.
	 */
	public function enqueue_styles( $hook_suffix ) {
		// Only load on our specific settings page
		if ( 'settings_page_' . ADCAVER_HELPER_SETTINGS_SLUG !== $hook_suffix ) {
			return;
		}

		$css_file_path = ADCAVER_HELPER_PLUGIN_DIR . 'admin/assets/css/admin.css';
		$css_version = file_exists($css_file_path) ? filemtime($css_file_path) : ADCAVER_HELPER_VERSION;

		wp_enqueue_style(
			'adcaver-helper-admin-css',
			ADCAVER_HELPER_PLUGIN_URL . 'admin/assets/css/admin.css',
			array(),
			$css_version,
			'all'
		);
	}

	/**
	 * Enqueues JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 * @param    string $hook_suffix The current admin page hook.
	 */
	public function enqueue_scripts( $hook_suffix ) {
		// Only load on our specific settings page
		if ( 'settings_page_' . ADCAVER_HELPER_SETTINGS_SLUG !== $hook_suffix ) {
			return;
		}

		$js_file_path = ADCAVER_HELPER_PLUGIN_DIR . 'admin/assets/js/admin.js';
		$js_version = file_exists($js_file_path) ? filemtime($js_file_path) : ADCAVER_HELPER_VERSION;

		wp_enqueue_script(
			'adcaver-helper-admin-js',
			ADCAVER_HELPER_PLUGIN_URL . 'admin/assets/js/admin.js',
			array( 'jquery' ),
			$js_version,
			true // Load in footer
		);
	}

    /**
	 * Adds a "Settings" link to the plugin action links on the plugins page.
	 *
	 * @since    1.0.0
	 * @param    array $links Existing action links for the plugin.
	 * @return   array Modified action links.
	 */
    public function add_settings_link( $links ) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            esc_url( admin_url( 'options-general.php?page=' . ADCAVER_HELPER_SETTINGS_SLUG ) ),
            esc_html__( 'Settings', 'adcaver-ad-helper' )
        );
        array_unshift( $links, $settings_link ); // Add to the beginning
        return $links;
    }
}