<?php
/**
 * Adcaver Ad Helper Loader
 *
 * Loads dependencies and registers hooks for the Adcaver Ad Helper plugin.
 *
 * @package    Adcaver_Ad_Helper
 * @subpackage Adcaver_Ad_Helper/includes
 * @author     Edu Brazeal, Adcaver Team <https://edubrazeal.com>
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

class Adcaver_Helper_Loader {

	/**
	 * A collection of actions to be registered with WordPress.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array    $actions    The actions registered with WordPress.
	 */
	protected $actions;

	/**
	 * A collection of filters to be registered with WordPress.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array    $filters    The filters registered with WordPress.
	 */
	protected $filters;

	/**
	 * Constructor: Initializes actions and filters, loads dependencies, and defines hooks.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		$this->actions = array();
		$this->filters = array();

		$this->load_dependencies();
		$this->define_admin_hooks();
		$this->define_frontend_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {
		require_once ADCAVER_HELPER_PLUGIN_DIR . 'admin/class-adcaver-admin.php';
		require_once ADCAVER_HELPER_PLUGIN_DIR . 'includes/class-adcaver-settings.php';
		require_once ADCAVER_HELPER_PLUGIN_DIR . 'includes/class-adcaver-frontend.php';
	}

	/**
	 * Register all hooks related to the admin area functionality.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {
		if ( ! is_admin() ) {
			return;
		}

		$plugin_admin   = new Adcaver_Helper_Admin();
		$plugin_settings = new Adcaver_Helper_Settings();

		$this->add_action( 'admin_menu', $plugin_admin, 'add_plugin_admin_menu' );
		$this->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		$this->add_action( 'admin_init', $plugin_settings, 'register_settings' );

		// Add link to settings page from plugins list
        $plugin_basename = ADCAVER_HELPER_PLUGIN_BASENAME;
        $this->add_filter( "plugin_action_links_$plugin_basename", $plugin_admin, 'add_settings_link' );
	}

	/**
	 * Register all hooks related to the public-facing functionality.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_frontend_hooks() {
		// Public hooks run even if is_admin() is true during AJAX calls.
		$plugin_frontend = new Adcaver_Helper_Frontend();

		$this->add_action( 'wp_head', $plugin_frontend, 'output_verification_meta_tag', 5 );
		$this->add_action( 'wp_enqueue_scripts', $plugin_frontend, 'enqueue_frontend_scripts' );
		// Register shortcodes on init
		$this->add_action( 'init', $plugin_frontend, 'register_ad_zone_shortcodes' );
		// Add filter for automatic ad insertion into content
		$this->add_filter( 'the_content', $plugin_frontend, 'auto_insert_ads_in_content' );
		// Note: The script_loader_tag filter is added conditionally inside enqueue_frontend_scripts
	}

	/**
	 * Add a new action to the collection.
	 *
	 * @since    1.0.0
	 * @param    string $hook          The name of the WordPress action.
	 * @param    object $component     A reference to the instance of the object on which the action is defined.
	 * @param    string $callback      The name of the method on the $component.
	 * @param    int    $priority      Optional. The priority. Default 10.
	 * @param    int    $accepted_args Optional. The number of arguments. Default 1.
	 */
	public function add_action( $hook, $component, $callback, $priority = 10, $accepted_args = 1 ) {
		$this->actions = $this->add( $this->actions, $hook, $component, $callback, $priority, $accepted_args );
	}

	/**
	 * Add a new filter to the collection.
	 *
	 * @since    1.0.0
	 * @param    string $hook          The name of the WordPress filter.
	 * @param    object $component     A reference to the instance of the object on which the filter is defined.
	 * @param    string $callback      The name of the method on the $component.
	 * @param    int    $priority      Optional. The priority. Default 10.
	 * @param    int    $accepted_args Optional. The number of arguments. Default 1.
	 */
	public function add_filter( $hook, $component, $callback, $priority = 10, $accepted_args = 1 ) {
		$this->filters = $this->add( $this->filters, $hook, $component, $callback, $priority, $accepted_args );
	}

	/**
	 * Utility function to register hooks into a collection.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @param    array  $hooks         The collection of hooks (actions or filters).
	 * @param    string $hook          The name of the WordPress hook.
	 * @param    object $component     The object instance.
	 * @param    string $callback      The callback method name.
	 * @param    int    $priority      The priority.
	 * @param    int    $accepted_args The number of arguments.
	 * @return   array                 The updated collection of hooks.
	 */
	private function add( $hooks, $hook, $component, $callback, $priority, $accepted_args ) {
		$hooks[] = array(
			'hook'          => $hook,
			'component'     => $component,
			'callback'      => $callback,
			'priority'      => $priority,
			'accepted_args' => $accepted_args,
		);
		return $hooks;
	}

	/**
	 * Register the filters and actions with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		foreach ( $this->filters as $hook ) {
			add_filter( $hook['hook'], array( $hook['component'], $hook['callback'] ), $hook['priority'], $hook['accepted_args'] );
		}
		foreach ( $this->actions as $hook ) {
			add_action( $hook['hook'], array( $hook['component'], $hook['callback'] ), $hook['priority'], $hook['accepted_args'] );
		}
	}

}