<?php
/**
 * Adcaver Ad Helper Settings
 *
 * Registers and renders the settings fields for the Adcaver Ad Helper plugin.
 *
 * @package    Adcaver_Ad_Helper
 * @subpackage Adcaver_Ad_Helper/includes
 * @author     Edu Brazeal, Adcaver Team <https://edubrazeal.com>
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

class Adcaver_Helper_Settings {

	/**
	 * Registers the settings sections and fields using the WordPress Settings API.
	 *
	 * @since    1.0.0
	 */
	public function register_settings() {
		// Register the main setting option group.
		register_setting(
			'adcaver_helper_options_group',
			ADCAVER_HELPER_OPTION_NAME,
			array( $this, 'sanitize_settings' )
		);

		// Section 1: Core Setup
		add_settings_section(
			'adcaver_helper_section_setup',
			__( 'Core Setup', 'adcaver-ad-helper' ),
			array( $this, 'render_setup_section_header' ),
			ADCAVER_HELPER_SETTINGS_SLUG
		);

		add_settings_field(
			'site_id',
			__( 'Your Adcaver Site ID', 'adcaver-ad-helper' ),
			array( $this, 'render_number_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_setup',
			array(
				'label_for'   => 'site_id',
				'description' => __( 'Enter the unique Site ID provided in your Adcaver Publisher Dashboard after adding this site.', 'adcaver-ad-helper' ),
				'required'    => true,
				'min'         => 1,
				'step'        => 1,
			)
		);

		add_settings_field(
			'verification_code',
			__( 'Verification Code', 'adcaver-ad-helper' ),
			array( $this, 'render_text_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_setup',
			array(
				'label_for'   => 'verification_code',
				'description' => __( 'Paste the verification meta tag code provided by Adcaver here. Only needed if your site status is "Pending Verification" on Adcaver.', 'adcaver-ad-helper' ),
				'required'    => false,
			)
		);

		// Section 2: Automatic Ad Placements for Posts
		add_settings_section(
			'adcaver_helper_section_auto_ads_posts',
			__( 'Automatic Ad Placements for Posts', 'adcaver-ad-helper' ),
			array( $this, 'render_auto_ads_posts_section_header' ),
			ADCAVER_HELPER_SETTINGS_SLUG
		);

		add_settings_field(
			'auto_ad_posts_before_content',
			__( 'Before Content', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_posts',
			array(
				'label_for'   => 'auto_ad_posts_before_content',
				'description' => __( 'Select ad type to display automatically at the very beginning of single post content.', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'auto_ad_posts_after_p1',
			__( 'After 1st Paragraph', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_posts',
			array(
				'label_for'   => 'auto_ad_posts_after_p1',
				'description' => __( 'Select ad type to display automatically after the first paragraph of single post content.', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'auto_ad_posts_after_p2',
			__( 'After 2nd Paragraph', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_posts',
			array(
				'label_for'   => 'auto_ad_posts_after_p2',
				'description' => __( 'Select ad type to display automatically after the second paragraph of single post content.', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'auto_ad_posts_after_p3',
			__( 'After 3rd Paragraph', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_posts',
			array(
				'label_for'   => 'auto_ad_posts_after_p3',
				'description' => __( 'Select ad type to display automatically after the third paragraph of single post content.', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'auto_ad_posts_mid_content',
			__( 'Mid-Content (50%)', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_posts',
			array(
				'label_for'   => 'auto_ad_posts_mid_content',
				'description' => __( 'Select ad type to display automatically at the middle of the post content (after 50% of paragraphs).', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'auto_ad_posts_after_content',
			__( 'After Content', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_posts',
			array(
				'label_for'   => 'auto_ad_posts_after_content',
				'description' => __( 'Select ad type to display automatically at the very end of single post content.', 'adcaver-ad-helper' ),
			)
		);

		// Section 3: Automatic Ad Placements for Pages
		add_settings_section(
			'adcaver_helper_section_auto_ads_pages',
			__( 'Automatic Ad Placements for Pages', 'adcaver-ad-helper' ),
			array( $this, 'render_auto_ads_pages_section_header' ),
			ADCAVER_HELPER_SETTINGS_SLUG
		);

		add_settings_field(
			'auto_ad_pages_before_content',
			__( 'Before Content', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_pages',
			array(
				'label_for'   => 'auto_ad_pages_before_content',
				'description' => __( 'Select ad type to display automatically at the very beginning of page content.', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'auto_ad_pages_after_content',
			__( 'After Content', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_pages',
			array(
				'label_for'   => 'auto_ad_pages_after_content',
				'description' => __( 'Select ad type to display automatically at the very end of page content.', 'adcaver-ad-helper' ),
			)
		);

		// Section 4: Homepage Specific Placements
		add_settings_section(
			'adcaver_helper_section_auto_ads_homepage',
			__( 'Homepage Specific Placements', 'adcaver-ad-helper' ),
			array( $this, 'render_auto_ads_homepage_section_header' ),
			ADCAVER_HELPER_SETTINGS_SLUG
		);

		add_settings_field(
			'auto_ad_homepage_before_content',
			__( 'Before Homepage Content', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_homepage',
			array(
				'label_for'   => 'auto_ad_homepage_before_content',
				'description' => __( 'Select ad type to display at the very beginning of your homepage content. Only appears on the front page.', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'auto_ad_homepage_after_content',
			__( 'After Homepage Content', 'adcaver-ad-helper' ),
			array( $this, 'render_ad_type_select_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_auto_ads_homepage',
			array(
				'label_for'   => 'auto_ad_homepage_after_content',
				'description' => __( 'Select ad type to display at the very end of your homepage content. Only appears on the front page.', 'adcaver-ad-helper' ),
			)
		);

		// Section 5: Overlay Ad Formats
		add_settings_section(
			'adcaver_helper_section_overlays',
			__( 'Overlay Ad Formats (ZealUp)', 'adcaver-ad-helper' ),
			array( $this, 'render_overlays_section_header' ),
			ADCAVER_HELPER_SETTINGS_SLUG
		);

		add_settings_field(
			'enable_vignette',
			__( 'Enable Vignette Ads', 'adcaver-ad-helper' ),
			array( $this, 'render_checkbox_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_overlays',
			array(
				'label_for'   => 'enable_vignette',
				'description' => __( 'Show full-screen ads during page navigations.', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'enable_anchor',
			__( 'Enable Anchor Ads', 'adcaver-ad-helper' ),
			array( $this, 'render_checkbox_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_overlays',
			array(
				'label_for'   => 'enable_anchor',
				'description' => __( 'Show a floating ad unit in the bottom corner of the screen.', 'adcaver-ad-helper' ),
			)
		);

		add_settings_field(
			'enable_sticky_bar',
			__( 'Enable Sticky Bar Ads', 'adcaver-ad-helper' ),
			array( $this, 'render_checkbox_field' ),
			ADCAVER_HELPER_SETTINGS_SLUG,
			'adcaver_helper_section_overlays',
			array(
				'label_for'   => 'enable_sticky_bar',
				'description' => __( 'Show a banner ad unit fixed to the bottom of the screen.', 'adcaver-ad-helper' ),
			)
		);

		// Section 6: Ad Zone Shortcodes (Informational)
        add_settings_section(
            'adcaver_helper_section_shortcodes',
            __( 'Ad Zone Shortcodes (Manual Placement)', 'adcaver-ad-helper' ),
            array( $this, 'render_shortcodes_section_header' ),
            ADCAVER_HELPER_SETTINGS_SLUG
        );
	}

	/**
	 * Sanitizes the settings array before saving.
	 *
	 * @since    1.0.0
	 * @param    array $input The raw input array from the form.
	 * @return   array The sanitized array.
	 */
	public function sanitize_settings( $input ) {
		$sanitized_input = array();
		$current_options = get_option( ADCAVER_HELPER_OPTION_NAME, array() );
		$allowed_ad_types = array( 'disabled', 'spotlight', 'banner', 'story' );

		$sanitized_input['site_id'] = isset( $input['site_id'] ) ? absint( $input['site_id'] ) : '';

		$sanitized_input['verification_code'] = isset( $input['verification_code'] ) ? trim( $input['verification_code'] ) : '';
        if ( preg_match( '/content="([^"]+)"/', $sanitized_input['verification_code'], $matches ) ) {
            $sanitized_input['verification_code'] = sanitize_text_field( $matches[1] );
        } else {
             $sanitized_input['verification_code'] = sanitize_text_field( $sanitized_input['verification_code'] );
        }

		// Sanitize Auto Placement Selectors for Posts
		$auto_placement_posts_keys = array(
			'auto_ad_posts_before_content',
			'auto_ad_posts_after_p1',
			'auto_ad_posts_after_p2',
			'auto_ad_posts_after_p3',
			'auto_ad_posts_mid_content',
			'auto_ad_posts_after_content',
		);
		foreach ( $auto_placement_posts_keys as $key ) {
			if ( isset( $input[ $key ] ) && in_array( $input[ $key ], $allowed_ad_types, true ) ) {
				$sanitized_input[ $key ] = $input[ $key ];
			} else {
				$sanitized_input[ $key ] = 'disabled';
			}
		}

		// Sanitize Auto Placement Selectors for Pages
		$auto_placement_pages_keys = array(
			'auto_ad_pages_before_content',
			'auto_ad_pages_after_content',
		);
		foreach ( $auto_placement_pages_keys as $key ) {
			if ( isset( $input[ $key ] ) && in_array( $input[ $key ], $allowed_ad_types, true ) ) {
				$sanitized_input[ $key ] = $input[ $key ];
			} else {
				$sanitized_input[ $key ] = 'disabled';
			}
		}

		// Sanitize Auto Placement Selectors for Homepage
		$auto_placement_homepage_keys = array(
			'auto_ad_homepage_before_content',
			'auto_ad_homepage_after_content',
		);
		foreach ( $auto_placement_homepage_keys as $key ) {
			if ( isset( $input[ $key ] ) && in_array( $input[ $key ], $allowed_ad_types, true ) ) {
				$sanitized_input[ $key ] = $input[ $key ];
			} else {
				$sanitized_input[ $key ] = 'disabled';
			}
		}

		// Sanitize Overlay Checkboxes
		$sanitized_input['enable_vignette'] = isset( $input['enable_vignette'] ) ? 1 : 0;
		$sanitized_input['enable_anchor'] = isset( $input['enable_anchor'] ) ? 1 : 0;
		$sanitized_input['enable_sticky_bar'] = isset( $input['enable_sticky_bar'] ) ? 1 : 0;

		return $sanitized_input;
	}

	/**
	 * Renders the header description for the Setup section.
	 *
	 * @since    1.0.0
	 */
	public function render_setup_section_header() {
		echo '<p>' . esc_html__( 'Connect your site to the Adcaver network.', 'adcaver-ad-helper' ) . '</p>';
	}

	/**
	 * Renders the header description for the Automatic Ad Placements for Posts section.
	 *
	 * @since    1.0.1
	 */
	public function render_auto_ads_posts_section_header() {
		echo '<p>' . esc_html__( 'Automatically insert ads into common locations within your single posts. Select the desired ad format for each location or leave as "Disabled".', 'adcaver-ad-helper' ) . '</p>';
	}

	/**
	 * Renders the header description for the Automatic Ad Placements for Pages section.
	 *
	 * @since    1.0.1
	 */
	public function render_auto_ads_pages_section_header() {
		echo '<p>' . esc_html__( 'Automatically insert ads into your pages. Pages typically have fewer placement options than posts to maintain a professional appearance.', 'adcaver-ad-helper' ) . '</p>';
	}

	/**
	 * Renders the header description for the Homepage Specific Placements section.
	 *
	 * @since    1.0.1
	 */
	public function render_auto_ads_homepage_section_header() {
		echo '<p>' . esc_html__( 'Control ad placements specifically for your homepage (front page). These settings only apply when viewing the homepage and will override other placement settings.', 'adcaver-ad-helper' ) . '</p>';
	}

	/**
	 * Renders the header description for the Overlay Ads section.
	 *
	 * @since    1.0.0
	 */
	public function render_overlays_section_header() {
		echo '<p>' . esc_html__( 'Enable or disable high-impact ad formats that display over your content. These require only your Site ID to be set.', 'adcaver-ad-helper' ) . '</p>';
	}

    /**
	 * Renders the header description for the Shortcodes section.
	 *
	 * @since    1.0.0
	 */
    public function render_shortcodes_section_header() {
        $options = get_option( ADCAVER_HELPER_OPTION_NAME );
		// Get the raw Site ID, or use a placeholder
        $site_id = ! empty( $options['site_id'] ) ? $options['site_id'] : '[YOUR_SITE_ID]';
		
        echo '<p>' . esc_html__( 'For manual ad placement within your content or theme files, copy and paste the relevant shortcode below. Replace [YOUR_SITE_ID] if you haven\'t saved your ID above.', 'adcaver-ad-helper' ) . '</p>';

        $shortcodes = array(
			// We escape $site_id here for display in the shortcode string
            'Spotlight (300x250)' => '[zam_spotlight_ad_zone site_id="' . esc_attr( $site_id ) . '"]',
            'Story (300x500) / App Install' => '[zam_story_ad_zone site_id="' . esc_attr( $site_id ) . '"]',
            'Banner (728x90)' => '[zam_banner_ad_zone site_id="' . esc_attr( $site_id ) . '"]',
            'HTML (300x250)' => '[zam_html_ad_zone site_id="' . esc_attr( $site_id ) . '"]',
        );

        echo '<table class="form-table adcaver-shortcode-table">';
        echo '<tbody>';
        foreach ($shortcodes as $label => $code) {
            echo '<tr>';
            echo '<th scope="row">' . esc_html($label) . '</th>';
			// $code is escaped here as it's placed in the 'value' attribute
            echo '<td><input type="text" class="regular-text readonly" value="' . esc_attr($code) . '" readonly onfocus="this.select();"></td>';
            echo '</tr>';
        }
         echo '<tr>';
         echo '<th scope="row">' . esc_html__('Specific Ad(s)', 'adcaver-ad-helper') . '</th>';
		 // This specific string is also escaped
		 $specific_ad_code = '[zam_spotlight_ad_zone site_id="' . esc_attr( $site_id ) . '" ad_id="123,456"]';
         echo '<td><input type="text" class="regular-text readonly" value="' . esc_attr( $specific_ad_code ) . '" readonly onfocus="this.select();"><p class="description">' . esc_html__('Optional: Add ad_id="YOUR_AD_ID(s)" to any zone to show specific ads.', 'adcaver-ad-helper') .'</p></td>';
         echo '</tr>';
        echo '</tbody>';
        echo '</table>';
        echo '<p class="description">' . esc_html__( 'These shortcodes require that your site is approved on Adcaver.com to work.', 'adcaver-ad-helper' ) . '</p>';

    }

	/**
	 * Renders a text input field for settings.
	 *
	 * @since    1.0.0
	 * @param    array $args Field arguments.
	 */
	public function render_text_field( $args ) {
		$options = get_option( ADCAVER_HELPER_OPTION_NAME );
		// Get raw value
		$value = isset( $options[ $args['label_for'] ] ) ? $options[ $args['label_for'] ] : '';
		$required = isset( $args['required'] ) && $args['required'];

		// Build input tag with attributes escaped on output
		echo '<input type="text" id="' . esc_attr( $args['label_for'] ) . '" name="' . esc_attr( ADCAVER_HELPER_OPTION_NAME ) . '[' . esc_attr( $args['label_for'] ) . ']" value="' . esc_attr( $value ) . '" class="regular-text"';
		
		if ( $required ) {
			echo ' required="required"'; // Static string, safe
		}
		
		echo ' />';

		if ( ! empty( $args['description'] ) ) {
			// This printf is fine as it uses esc_html
			printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
		}
	}

	/**
	 * Renders a number input field for settings.
	 *
	 * @since    1.0.0
	 * @param    array $args Field arguments.
	 */
	public function render_number_field( $args ) {
		$options = get_option( ADCAVER_HELPER_OPTION_NAME );
		// Get raw values
		$value = isset( $options[ $args['label_for'] ] ) ? $options[ $args['label_for'] ] : '';
		$required = isset( $args['required'] ) && $args['required'];
		$min = isset( $args['min'] ) ? $args['min'] : null;
		$step = isset( $args['step'] ) ? $args['step'] : null;

		// Build input tag with attributes escaped on output
		echo '<input type="number" id="' . esc_attr( $args['label_for'] ) . '" name="' . esc_attr( ADCAVER_HELPER_OPTION_NAME ) . '[' . esc_attr( $args['label_for'] ) . ']" value="' . esc_attr( $value ) . '" class="regular-text"';

		if ( $required ) {
			echo ' required="required"'; // Static string, safe
		}
		if ( ! is_null( $min ) ) {
			echo ' min="' . esc_attr( $min ) . '"'; // Escaped value
		}
		if ( ! is_null( $step ) ) {
			echo ' step="' . esc_attr( $step ) . '"'; // Escaped value
		}

		echo ' />';

		if ( ! empty( $args['description'] ) ) {
			printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
		}
	}

	/**
	 * Renders a checkbox field for settings.
	 *
	 * @since    1.0.0
	 * @param    array $args Field arguments.
	 */
	public function render_checkbox_field( $args ) {
		$options = get_option( ADCAVER_HELPER_OPTION_NAME );
		// Get raw value
		$checked_status = isset( $options[ $args['label_for'] ] ) ? $options[ $args['label_for'] ] : 0;

		// Build input tag
		echo '<input type="checkbox" id="' . esc_attr( $args['label_for'] ) . '" name="' . esc_attr( ADCAVER_HELPER_OPTION_NAME ) . '[' . esc_attr( $args['label_for'] ) . ']" value="1"';

		// Use WordPress's checked() function, which safely *returns* the attribute string
		echo ' ' . checked( 1, $checked_status, false );

		echo ' />';

		if ( ! empty( $args['description'] ) ) {
			// This printf is fine
			printf( '<label for="%1$s"> <span class="description">%2$s</span></label>', esc_attr( $args['label_for'] ), esc_html( $args['description'] ) );
		}
	}

	/**
	 * Renders a select dropdown field for choosing ad types.
	 *
	 * @since    1.0.1
	 * @param    array $args Field arguments.
	 */
	public function render_ad_type_select_field( $args ) {
		$options = get_option( ADCAVER_HELPER_OPTION_NAME );
		$current_value = isset( $options[ $args['label_for'] ] ) ? $options[ $args['label_for'] ] : 'disabled';

		$ad_types = array(
			'disabled'  => __( 'Disabled', 'adcaver-ad-helper' ),
			'spotlight' => __( 'Spotlight Ad (300x250)', 'adcaver-ad-helper' ),
			'banner'    => __( 'Banner Ad (728x90)', 'adcaver-ad-helper' ),
			'story'     => __( 'Story Ad (300x500)', 'adcaver-ad-helper' ),
		);

		printf( '<select id="%1$s" name="%2$s[%1$s]">',
			esc_attr( $args['label_for'] ),
			esc_attr( ADCAVER_HELPER_OPTION_NAME )
		);

		foreach ( $ad_types as $value => $label ) {
			printf( '<option value="%s"%s>%s</option>',
				esc_attr( $value ),
				selected( $current_value, $value, false ), // selected() is safe
				esc_html( $label )
			);
		}

		echo '</select>';

		if ( ! empty( $args['description'] ) ) {
			printf( '<p class="description">%s</p>', esc_html( $args['description'] ) );
		}
	}
}